<?php
/**
 * Admin Program Profile Page
 * View and manage program details
 */

session_start();

// Database connection
$host = 'localhost';
$dbname = 'jv_overseas';
$username = 'root';
$password = '';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    die("Connection failed: " . $e->getMessage());
}

// Authentication check
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'admin') {
    header('Location: admin-login.php');
    exit;
}

// Get admin data
$stmt = $pdo->prepare("SELECT a.*, u.email FROM admins a JOIN users u ON a.id = u.id WHERE a.id = ?");
$stmt->execute([$_SESSION['user_id']]);
$admin = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$admin) {
    die("Admin profile not found.");
}

// Get program ID from URL
$program_id = $_GET['id'] ?? null;
if (!$program_id) {
    header('Location: admin-programs.php');
    exit;
}

// Get program data
$stmt = $pdo->prepare("SELECT p.*, u.name as university_name, u.id as university_id, u.logo as university_logo,
                      co.name as country_name, co.code as country_code
                      FROM programs p
                      JOIN universities u ON p.university_id = u.id
                      JOIN countries co ON u.country_id = co.id
                      WHERE p.id = ?");
$stmt->execute([$program_id]);
$program = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$program) {
    die("Program not found.");
}

// Initialize arrays to prevent errors if tables don't exist
$requirements = [];
$documents = [];
$deadlines = [];

// Safely get program admission requirements (only if table exists)
try {
    $requirements_stmt = $pdo->prepare("SELECT * FROM admission_requirements 
                                      WHERE program_id = ? 
                                      ORDER BY created_at DESC");
    $requirements_stmt->execute([$program_id]);
    $requirements = $requirements_stmt->fetchAll();
} catch (PDOException $e) {
    // Table doesn't exist, just continue with empty array
    error_log("Table admission_requirements not found: " . $e->getMessage());
}

// Safely get program documents (only if table exists)
try {
    $documents_stmt = $pdo->prepare("SELECT * FROM program_documents 
                                   WHERE program_id = ? 
                                   ORDER BY created_at DESC");
    $documents_stmt->execute([$program_id]);
    $documents = $documents_stmt->fetchAll();
} catch (PDOException $e) {
    // Table doesn't exist, just continue with empty array
    error_log("Table program_documents not found: " . $e->getMessage());
}

// Safely get program deadlines (only if table exists)
try {
    $deadlines_stmt = $pdo->prepare("SELECT * FROM application_deadlines 
                                   WHERE program_id = ? 
                                   ORDER BY deadline_date ASC");
    $deadlines_stmt->execute([$program_id]);
    $deadlines = $deadlines_stmt->fetchAll();
} catch (PDOException $e) {
    // Table doesn't exist, just continue with empty array
    error_log("Table application_deadlines not found: " . $e->getMessage());
}

// Get program statistics - Fixed the error by separating prepare and execute
$stats = [
    'total_applications' => 0,
    'accepted_applications' => 0,
    'rejected_applications' => 0,
];

try {
    // Fixed: Separate prepare and execute steps
    $total_applications_stmt = $pdo->prepare("SELECT COUNT(*) FROM applications WHERE program_id = ?");
    $total_applications_stmt->execute([$program_id]);
    $stats['total_applications'] = $total_applications_stmt->fetchColumn();
    
    $accepted_applications_stmt = $pdo->prepare("SELECT COUNT(*) FROM applications WHERE program_id = ? AND application_status = 'accepted'");
    $accepted_applications_stmt->execute([$program_id]);
    $stats['accepted_applications'] = $accepted_applications_stmt->fetchColumn();
    
    $rejected_applications_stmt = $pdo->prepare("SELECT COUNT(*) FROM applications WHERE program_id = ? AND application_status = 'rejected'");
    $rejected_applications_stmt->execute([$program_id]);
    $stats['rejected_applications'] = $rejected_applications_stmt->fetchColumn();
} catch (PDOException $e) {
    error_log("Error getting application stats: " . $e->getMessage());
}

// Handle form submissions
$error = '';
$success = '';

// Update program details
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['update_program'])) {
    $name = trim($_POST['name'] ?? '');
    $degree_level = $_POST['degree_level'] ?? '';
    $duration_months = (int)($_POST['duration_months'] ?? 0);
    $tuition_fee = (float)($_POST['tuition_fee'] ?? 0);
    $tuition_fee_currency = $_POST['tuition_fee_currency'] ?? 'USD';
    $language_requirements = trim($_POST['language_requirements'] ?? '');
    $academic_requirements = trim($_POST['academic_requirements'] ?? '');
    $description = trim($_POST['description'] ?? '');
    
    if (empty($name)) {
        $error = "Program name is required.";
    } elseif (!in_array($degree_level, ['diploma', 'bachelors', 'masters', 'phd', 'certificate'])) {
        $error = "Invalid degree level selected.";
    } elseif ($duration_months <= 0) {
        $error = "Duration must be greater than 0 months.";
    } elseif ($tuition_fee < 0) {
        $error = "Tuition fee cannot be negative.";
    } else {
        try {
            $stmt = $pdo->prepare("UPDATE programs SET 
                                  name = ?, degree_level = ?, duration_months = ?, 
                                  tuition_fee = ?, tuition_fee_currency = ?,
                                  language_requirements = ?, academic_requirements = ?, 
                                  description = ?, updated_at = NOW()
                                  WHERE id = ?");
            $stmt->execute([
                $name, $degree_level, $duration_months,
                $tuition_fee, $tuition_fee_currency,
                $language_requirements, $academic_requirements,
                $description, $program_id
            ]);
            
            // Log the activity
            $pdo->prepare("INSERT INTO activities (user_id, activity_type, description, created_at) 
                          VALUES (?, ?, ?, NOW())")->execute([
                $_SESSION['user_id'],
                "program_updated",
                "Admin {$admin['first_name']} {$admin['last_name']} updated program {$program_id}: {$name}"
            ]);
            
            $success = "Program details updated successfully!";
            
            // Refresh program data
            $stmt = $pdo->prepare("SELECT p.*, u.name as university_name, u.id as university_id, u.logo as university_logo,
                                  co.name as country_name, co.code as country_code
                                  FROM programs p
                                  JOIN universities u ON p.university_id = u.id
                                  JOIN countries co ON u.country_id = co.id
                                  WHERE p.id = ?");
            $stmt->execute([$program_id]);
            $program = $stmt->fetch(PDO::FETCH_ASSOC);
            
        } catch (Exception $e) {
            $error = "Failed to update program details. Please try again.";
        }
    }
}

// Logout
if (isset($_GET['logout'])) {
    session_destroy();
    header('Location: admin-login.php');
    exit;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?= htmlspecialchars($program['name']) ?> - JV Overseas Education</title>
    <meta content="width=device-width, initial-scale=1.0" name="viewport">
    <link rel="preconnect" href="https://fonts.gstatic.com">
    <link href="https://fonts.googleapis.com/css2?family=Jost:wght@500;600;700&family=Open+Sans:wght@400;600&display=swap" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.10.0/css/all.min.css" rel="stylesheet">
    <link href="lib/owlcarousel/assets/owl.carousel.min.css" rel="stylesheet">
    <link href="css/style.css" rel="stylesheet">
    <style>
        .program-header {
            background: linear-gradient(135deg, #3fb7fe, #1a73e8);
            color: white;
            border-radius: 10px 10px 0 0;
            padding: 30px;
            margin-bottom: 30px;
        }
        .university-logo {
            width: 80px;
            height: 80px;
            object-fit: contain;
            background: white;
            padding: 5px;
            border-radius: 5px;
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        }
        .section-title {
            position: relative;
            padding-bottom: 10px;
            margin-bottom: 25px;
            border-bottom: 2px solid #3fb7fe;
        }
        .section-title::after {
            content: '';
            position: absolute;
            bottom: -2px;
            left: 0;
            width: 60px;
            height: 2px;
            background: #3fb7fe;
        }
        .info-card {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .info-item {
            margin-bottom: 15px;
            padding-bottom: 15px;
            border-bottom: 1px dashed #eee;
        }
        .info-item:last-child {
            border-bottom: none;
            margin-bottom: 0;
            padding-bottom: 0;
        }
        .info-label {
            font-weight: 600;
            color: #333;
            margin-bottom: 5px;
        }
        .level-badge {
            font-size: 0.8em;
            padding: 5px 12px;
            border-radius: 20px;
            font-weight: 600;
        }
        .level-diploma {
            background: #fff3cd;
            color: #856404;
        }
        .level-bachelors {
            background: #d1ecf1;
            color: #0c5460;
        }
        .level-masters {
            background: #d4edda;
            color: #155724;
        }
        .level-phd {
            background: #f8d7da;
            color: #721c24;
        }
        .level-certificate {
            background: #e9ecef;
            color: #495057;
        }
        .requirement-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 10px 0;
            border-bottom: 1px solid #eee;
        }
        .requirement-item:last-child {
            border-bottom: none;
        }
        .info-unavailable {
            color: #6c757d;
            font-style: italic;
        }
        .action-buttons {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
        }
        .modal-dialog {
            max-width: 600px;
        }
        .stats-card {
            border-left: 5px solid #3fb7fe;
            margin-bottom: 20px;
        }
        .stats-card .icon {
            font-size: 2em;
            opacity: 0.2;
        }
        .application-status {
            font-size: 0.8em;
            padding: 5px 12px;
            border-radius: 20px;
            font-weight: 600;
        }
        .application-status-accepted {
            background: #d4edda;
            color: #155724;
        }
        .application-status-rejected {
            background: #f8d7da;
            color: #721c24;
        }
        .tab-content {
            padding: 20px 0;
        }
    </style>
</head>
<body>

<?php include 'admin-navbar.php'; ?>

<div class="container py-5">
    <!-- Program Header -->
    <div class="program-header">
        <div class="row align-items-center">
            <div class="col-md-2 text-center mb-3 mb-md-0">
                <img src="<?= htmlspecialchars($program['university_logo'] ?: 'img/university-placeholder.png') ?>" 
                     alt="<?= htmlspecialchars($program['university_name']) ?>" class="university-logo">
            </div>
            <div class="col-md-8">
                <h1><?= htmlspecialchars($program['name']) ?></h1>
                <p class="mb-1"><strong><?= htmlspecialchars($program['university_name']) ?></strong></p>
                <p><i class="fas fa-map-marker-alt"></i> <?= htmlspecialchars($program['country_name']) ?></p>
            </div>
            <div class="col-md-2 text-md-right">
                <span class="level-badge level-<?= $program['degree_level'] ?>">
                    <?= ucfirst($program['degree_level']) ?>
                </span>
                <p class="mt-2">
                    <strong>$<?= number_format($program['tuition_fee']) ?></strong><br>
                    <small><?= $program['duration_months'] ?> months</small>
                </p>
            </div>
        </div>
    </div>

    <?php if ($error): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
    <?php endif; ?>
    
    <?php if ($success): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
    <?php endif; ?>

    <!-- Tabs -->
    <ul class="nav nav-pills mb-4" id="programTabs" role="tablist">
        <li class="nav-item">
            <a class="nav-link active" id="overview-tab" data-toggle="pill" href="#overview" role="tab">Overview</a>
        </li>
        <li class="nav-item">
            <a class="nav-link" id="requirements-tab" data-toggle="pill" href="#requirements" role="tab">Admission Requirements</a>
        </li>
        <li class="nav-item">
            <a class="nav-link" id="documents-tab" data-toggle="pill" href="#documents" role="tab">Required Documents</a>
        </li>
        <li class="nav-item">
            <a class="nav-link" id="deadlines-tab" data-toggle="pill" href="#deadlines" role="tab">Application Deadlines</a>
        </li>
        <li class="nav-item">
            <a class="nav-link" id="applications-tab" data-toggle="pill" href="#applications" role="tab">Applications</a>
        </li>
    </ul>

    <div class="tab-content">
        <!-- Overview Tab -->
        <div class="tab-pane fade show active" id="overview" role="tabpanel">
            <div class="action-buttons">
                <button type="button" class="btn btn-primary" data-toggle="modal" data-target="#editProgramModal">
                    <i class="fas fa-edit"></i> Edit Program
                </button>
                <a href="admin-university-profile.php?id=<?= $program['university_id'] ?>" class="btn btn-outline-primary">
                    <i class="fas fa-building"></i> View University
                </a>
            </div>
            
            <div class="row">
                <div class="col-lg-8">
                    <h5 class="section-title">Program Details</h5>
                    <div class="info-card">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="info-item">
                                    <div class="info-label">Program Name</div>
                                    <div><?= htmlspecialchars($program['name']) ?></div>
                                </div>
                                <div class="info-item">
                                    <div class="info-label">Degree Level</div>
                                    <div>
                                        <span class="level-badge level-<?= $program['degree_level'] ?>">
                                            <?= ucfirst($program['degree_level']) ?>
                                        </span>
                                    </div>
                                </div>
                                <div class="info-item">
                                    <div class="info-label">Duration</div>
                                    <div><?= $program['duration_months'] ?> months</div>
                                </div>
                                <div class="info-item">
                                    <div class="info-label">Tuition Fee</div>
                                    <div><?= $program['tuition_fee_currency'] ?> <?= number_format($program['tuition_fee'], 2) ?></div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="info-item">
                                    <div class="info-label">Language Requirements</div>
                                    <div><?= htmlspecialchars($program['language_requirements'] ?? 'Not specified') ?></div>
                                </div>
                                <div class="info-item">
                                    <div class="info-label">Academic Requirements</div>
                                    <div><?= htmlspecialchars($program['academic_requirements'] ?? 'Not specified') ?></div>
                                </div>
                                <div class="info-item">
                                    <div class="info-label">Country</div>
                                    <div><?= htmlspecialchars($program['country_name']) ?></div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <h5 class="section-title mt-4">Program Description</h5>
                    <div class="info-card">
                        <p><?= nl2br(htmlspecialchars($program['description'] ?? 'No description available.')) ?></p>
                    </div>
                </div>
                
                <div class="col-lg-4">
                    <h5 class="section-title">Statistics</h5>
                    <div class="stats-card">
                        <div class="card-body">
                            <div class="float-right"><i class="fas fa-file-alt icon"></i></div>
                            <h6>Total Applications</h6>
                            <h3><?= $stats['total_applications'] ?></h3>
                        </div>
                    </div>
                    <div class="stats-card" style="border-left-color: #28a745;">
                        <div class="card-body">
                            <div class="float-right"><i class="fas fa-check-circle icon" style="color: #28a745;"></i></div>
                            <h6>Accepted</h6>
                            <h3><?= $stats['accepted_applications'] ?></h3>
                        </div>
                    </div>
                    <div class="stats-card" style="border-left-color: #dc3545;">
                        <div class="card-body">
                            <div class="float-right"><i class="fas fa-times-circle icon" style="color: #dc3545;"></i></div>
                            <h6>Rejected</h6>
                            <h3><?= $stats['rejected_applications'] ?></h3>
                        </div>
                    </div>
                    
                    <h5 class="section-title mt-4">Quick Actions</h5>
                    <div class="list-group">
                        <a href="admin-applications.php?program_id=<?= $program_id ?>" class="list-group-item list-group-item-action">
                            <i class="fas fa-file-alt mr-2"></i> View Applications
                        </a>
                        <a href="admin-university-profile.php?id=<?= $program['university_id'] ?>" class="list-group-item list-group-item-action">
                            <i class="fas fa-building mr-2"></i> View University
                        </a>
                        <a href="admin-programs.php" class="list-group-item list-group-item-action">
                            <i class="fas fa-arrow-left mr-2"></i> Back to Programs
                        </a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Requirements Tab -->
        <div class="tab-pane fade" id="requirements" role="tabpanel">
            <div class="alert alert-info">
                <i class="fas fa-info-circle"></i> The admission requirements feature is not yet implemented in the database.
            </div>
            
            <div class="info-card">
                <p class="info-unavailable">Admission requirements for this program are not available in the current system.</p>
                <p>The following information is based on the program's general requirements:</p>
                
                <div class="form-group">
                    <label>Language Requirements</label>
                    <p><?= htmlspecialchars($program['language_requirements'] ?? 'Not specified') ?></p>
                </div>
                
                <div class="form-group">
                    <label>Academic Requirements</label>
                    <p><?= htmlspecialchars($program['academic_requirements'] ?? 'Not specified') ?></p>
                </div>
            </div>
        </div>

        <!-- Documents Tab -->
        <div class="tab-pane fade" id="documents" role="tabpanel">
            <div class="alert alert-info">
                <i class="fas fa-info-circle"></i> The program documents feature is not yet implemented in the database.
            </div>
            
            <div class="info-card">
                <p class="info-unavailable">Required documents for this program are not available in the current system.</p>
                <p>Typically, students need to submit the following documents:</p>
                <ul>
                    <li>Academic transcripts</li>
                    <li>Passport copy</li>
                    <li>Language proficiency test results</li>
                    <li>Statement of purpose</li>
                    <li>Letters of recommendation</li>
                    <li>Financial statements</li>
                </ul>
                <p>Please check the university's official website for specific requirements.</p>
            </div>
        </div>

        <!-- Deadlines Tab -->
        <div class="tab-pane fade" id="deadlines" role="tabpanel">
            <div class="alert alert-info">
                <i class="fas fa-info-circle"></i> The application deadlines feature is not yet implemented in the database.
            </div>
            
            <div class="info-card">
                <p class="info-unavailable">Application deadlines for this program are not available in the current system.</p>
                <p>Application periods typically follow these patterns:</p>
                
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Intake Period</th>
                                <th>Application Window</th>
                                <th>Decision Timeline</th>
                            </tr>
                        </thead>
                        <tbody>
                            <tr>
                                <td>Fall Intake</td>
                                <td>Sep - Jan</td>
                                <td>Dec - Apr</td>
                            </tr>
                            <tr>
                                <td>Spring Intake</td>
                                <td>Mar - Jul</td>
                                <td>Jun - Aug</td>
                            </tr>
                        </tbody>
                    </table>
                </div>
                
                <p><strong>Note:</strong> Deadlines vary by university and program. Please verify with the specific institution.</p>
            </div>
        </div>

        <!-- Applications Tab -->
        <div class="tab-pane fade" id="applications" role="tabpanel">
            <div class="action-buttons">
                <a href="admin-applications.php?program_id=<?= $program_id ?>" class="btn btn-primary">
                    <i class="fas fa-list"></i> View All Applications
                </a>
            </div>
            
            <?php 
            // Get recent applications
            try {
                $recent_applications = $pdo->prepare("SELECT a.*, 
                                                    s.first_name, s.last_name, s.email,
                                                    a.application_status
                                            FROM applications a
                                            JOIN students s ON a.student_id = s.id
                                            WHERE a.program_id = ?
                                            ORDER BY a.created_at DESC LIMIT 5");
                $recent_applications->execute([$program_id]);
                $recent_applications = $recent_applications->fetchAll();
            } catch (PDOException $e) {
                error_log("Error fetching applications: " . $e->getMessage());
                $recent_applications = [];
            }
            ?>
            
            <?php if (empty($recent_applications)): ?>
                <div class="text-center py-5">
                    <i class="fas fa-file-alt fa-3x text-muted mb-3"></i>
                    <p>No applications received yet.</p>
                </div>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Student</th>
                                <th>Email</th>
                                <th>Status</th>
                                <th>Submitted</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($recent_applications as $app): ?>
                            <tr>
                                <td><?= htmlspecialchars($app['first_name'] . ' ' . $app['last_name']) ?></td>
                                <td><?= htmlspecialchars($app['email']) ?></td>
                                <td>
                                    <span class="application-status application-status-<?= $app['application_status'] ?>">
                                        <?= ucfirst($app['application_status']) ?>
                                    </span>
                                </td>
                                <td><?= date('M j, Y', strtotime($app['created_at'])) ?></td>
                                <td>
                                    <a href="admin-application-review.php?id=<?= $app['id'] ?>" class="btn btn-sm btn-primary">Review</a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>
</div>

<!-- Edit Program Modal -->
<div class="modal fade" id="editProgramModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <form method="POST">
                <div class="modal-header">
                    <h5 class="modal-title">Edit Program: <?= htmlspecialchars($program['name']) ?></h5>
                    <button type="button" class="close" data-dismiss="modal">
                        <span>&times;</span>
                    </button>
                </div>
                <div class="modal-body">
                    <div class="form-group">
                        <label>Program Name *</label>
                        <input type="text" name="name" class="form-control" value="<?= htmlspecialchars($program['name']) ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Degree Level *</label>
                        <select name="degree_level" class="form-control" required>
                            <option value="diploma" <?= $program['degree_level'] === 'diploma' ? 'selected' : '' ?>>Diploma</option>
                            <option value="bachelors" <?= $program['degree_level'] === 'bachelors' ? 'selected' : '' ?>>Bachelor's</option>
                            <option value="masters" <?= $program['degree_level'] === 'masters' ? 'selected' : '' ?>>Master's</option>
                            <option value="phd" <?= $program['degree_level'] === 'phd' ? 'selected' : '' ?>>PhD</option>
                            <option value="certificate" <?= $program['degree_level'] === 'certificate' ? 'selected' : '' ?>>Certificate</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Duration (months) *</label>
                        <input type="number" name="duration_months" class="form-control" value="<?= $program['duration_months'] ?>" min="1" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Tuition Fee *</label>
                        <div class="input-group">
                            <input type="number" name="tuition_fee" class="form-control" value="<?= $program['tuition_fee'] ?>" min="0" step="0.01" required>
                            <div class="input-group-append">
                                <select name="tuition_fee_currency" class="form-control">
                                    <option value="USD" <?= $program['tuition_fee_currency'] === 'USD' ? 'selected' : '' ?>>USD</option>
                                    <option value="EUR" <?= $program['tuition_fee_currency'] === 'EUR' ? 'selected' : '' ?>>EUR</option>
                                    <option value="GBP" <?= $program['tuition_fee_currency'] === 'GBP' ? 'selected' : '' ?>>GBP</option>
                                    <option value="UGX" <?= $program['tuition_fee_currency'] === 'UGX' ? 'selected' : '' ?>>UGX</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Language Requirements</label>
                        <input type="text" name="language_requirements" class="form-control" value="<?= htmlspecialchars($program['language_requirements'] ?? '') ?>">
                    </div>
                    
                    <div class="form-group">
                        <label>Academic Requirements</label>
                        <input type="text" name="academic_requirements" class="form-control" value="<?= htmlspecialchars($program['academic_requirements'] ?? '') ?>">
                    </div>
                    
                    <div class="form-group">
                        <label>Description</label>
                        <textarea name="description" class="form-control" rows="4" placeholder="Enter program description..."><?= htmlspecialchars($program['description'] ?? '') ?></textarea>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                    <button type="submit" name="update_program" class="btn btn-primary">Update Program</button>
                </div>
            </form>
        </div>
    </div>
</div>

<?php include 'footer.php'; ?>
<script src="https://code.jquery.com/jquery-3.4.1.min.js"></script>
<script src="https://stackpath.bootstrapcdn.com/bootstrap/4.4.1/js/bootstrap.bundle.min.js"></script>
<script>
    // Form submission confirmation
    $('form').on('submit', function(e) {
        if ($(this).find('[name="update_program"]').length) {
            if (!confirm('Update program details?')) {
                e.preventDefault();
            }
        }
    });
</script>
</body>
</html>