<?php
require_once __DIR__ . '/../../includes/config.php';
require_once '../../includes/auth.php';
require_once '../../includes/dashboard-config.php';

// Check if user is logged in and is a student
if (!isLoggedIn() || $_SESSION['user_type'] != 'student') {
    header('Location: ../login.php');
    exit();
}

// Get student data
$student_id = $_SESSION['user_id'];
$stmt = $pdo->prepare("SELECT * FROM students WHERE id = ?");
$stmt->execute([$student_id]);
$student = $stmt->fetch(PDO::FETCH_ASSOC);

// Get application status
$stmt = $pdo->prepare("SELECT status FROM applications WHERE student_id = ? ORDER BY created_at DESC LIMIT 1");
$stmt->execute([$student_id]);
$application = $stmt->fetch(PDO::FETCH_ASSOC);

// Get upcoming tasks
$stmt = $pdo->prepare("SELECT * FROM tasks WHERE student_id = ? AND completed = 0 AND deadline >= CURDATE() ORDER BY deadline ASC LIMIT 5");
$stmt->execute([$student_id]);
$tasks = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Get university choices
$stmt = $pdo->prepare("SELECT u.name, u.logo, a.status 
                      FROM university_applications a 
                      JOIN universities u ON a.university_id = u.id 
                      WHERE a.student_id = ?");
$stmt->execute([$student_id]);
$universities = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>JV Overseas Education - Student Dashboard</title>
    <meta content="width=device-width, initial-scale=1.0" name="viewport">
    <!-- Include your existing CSS files -->
    <link href="../../css/style.css" rel="stylesheet">
    <!-- Dashboard-specific CSS -->
    <link href="../../css/dashboard.css" rel="stylesheet">
    <style>
        /* Dashboard-specific styles that match your branding */
        .dashboard-header {
            background: linear-gradient(rgba(63, 183, 254, 0.9), rgba(63, 183, 254, 0.9));
            color: white;
        }
        .progress-tracker .active {
            background-color: #3fb7fe;
            color: white;
        }
        .task-alert {
            border-left: 4px solid #3fb7fe;
        }
        .university-card {
            transition: all 0.3s;
        }
        .university-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 20px rgba(0,0,0,0.1);
        }
    </style>
</head>
<body>
    <!-- Dashboard Header -->
    <header class="dashboard-header py-3">
        <div class="container">
            <div class="d-flex justify-content-between align-items-center">
                <div class="d-flex align-items-center">
                    <img src="../../img/logo-white.png" alt="JV Overseas" style="height: 40px;">
                    <h4 class="mb-0 ml-3">Student Dashboard</h4>
                </div>
                <div class="dropdown">
                    <button class="btn btn-light dropdown-toggle" type="button" id="profileDropdown" data-toggle="dropdown">
                        <img src="<?= !empty($student['profile_photo']) ? '../../uploads/'.$student['profile_photo'] : '../../img/default-profile.jpg' ?>" 
                             class="rounded-circle mr-2" width="30" height="30">
                        <?= htmlspecialchars($student['first_name']) ?>
                    </button>
                    <div class="dropdown-menu dropdown-menu-right">
                        <a class="dropdown-item" href="profile.php"><i class="fas fa-user mr-2"></i>Profile</a>
                        <a class="dropdown-item" href="../logout.php"><i class="fas fa-sign-out-alt mr-2"></i>Logout</a>
                    </div>
                </div>
            </div>
        </div>
    </header>

    <!-- Dashboard Navigation -->
    <nav class="navbar navbar-expand-lg navbar-light bg-white shadow-sm">
        <div class="container">
            <button class="navbar-toggler" type="button" data-toggle="collapse" data-target="#dashboardNav">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="dashboardNav">
                <ul class="navbar-nav">
                    <li class="nav-item active">
                        <a class="nav-link" href="index.php"><i class="fas fa-home mr-1"></i> Dashboard</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="applications.php"><i class="fas fa-file-alt mr-1"></i> Applications</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="documents.php"><i class="fas fa-file-upload mr-1"></i> Documents</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="payments.php"><i class="fas fa-money-bill-wave mr-1"></i> Payments</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="messages.php"><i class="fas fa-envelope mr-1"></i> Messages</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="appointments.php"><i class="fas fa-calendar-check mr-1"></i> Appointments</a>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <!-- Main Dashboard Content -->
    <div class="container py-5">
        <div class="row">
            <!-- Welcome Section -->
            <div class="col-lg-8">
                <div class="card mb-4 shadow-sm">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <img src="<?= !empty($student['profile_photo']) ? '../../uploads/'.$student['profile_photo'] : '../../img/default-profile.jpg' ?>" 
                                 class="rounded-circle mr-4" width="80" height="80">
                            <div>
                                <h2 class="mb-1">Welcome back, <?= htmlspecialchars($student['first_name']) ?>!</h2>
                                <p class="text-muted mb-0">Last login: <?= date('M j, Y g:i a', strtotime($student['last_login'])) ?></p>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Application Status -->
                <div class="card mb-4 shadow-sm">
                    <div class="card-body">
                        <h4 class="card-title mb-4">Your Application Status</h4>
                        <div class="progress-tracker">
                            <?php
                            $stages = ['Not Started', 'Profile Created', 'Documents Uploaded', 'Applications Submitted', 'Interviews Completed', 'Admitted'];
                            $current_stage = $application ? array_search($application['status'], $stages) : 0;
                            
                            foreach ($stages as $index => $stage) {
                                $active = $index <= $current_stage ? 'active' : '';
                                echo "<div class='stage {$active}'>
                                        <div class='stage-circle'>{$index}</div>
                                        <div class='stage-label'>{$stage}</div>
                                      </div>";
                            }
                            ?>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Upcoming Tasks -->
            <div class="col-lg-4">
                <div class="card mb-4 shadow-sm">
                    <div class="card-body">
                        <h4 class="card-title mb-4">Upcoming Tasks</h4>
                        <?php if (count($tasks) > 0): ?>
                            <ul class="list-group list-group-flush">
                                <?php foreach ($tasks as $task): ?>
                                    <li class="list-group-item task-alert">
                                        <div class="d-flex justify-content-between">
                                            <div>
                                                <h6 class="mb-1"><?= htmlspecialchars($task['title']) ?></h6>
                                                <small class="text-muted">Due: <?= date('M j, Y', strtotime($task['deadline'])) ?></small>
                                            </div>
                                            <a href="#" class="btn btn-sm btn-outline-primary">View</a>
                                        </div>
                                    </li>
                                <?php endforeach; ?>
                            </ul>
                        <?php else: ?>
                            <p class="text-muted">No upcoming tasks at the moment.</p>
                        <?php endif; ?>
                        <a href="applications.php" class="btn btn-primary btn-block mt-3">View All Tasks</a>
                    </div>
                </div>
            </div>
        </div>

        <!-- University Choices -->
        <div class="row mt-4">
            <div class="col-12">
                <div class="card shadow-sm">
                    <div class="card-body">
                        <h4 class="card-title mb-4">Your University Choices</h4>
                        <div class="row">
                            <?php if (count($universities) > 0): ?>
                                <?php foreach ($universities as $uni): ?>
                                    <div class="col-md-4 mb-4">
                                        <div class="card university-card h-100">
                                            <div class="card-body text-center">
                                                <img src="../../uploads/universities/<?= htmlspecialchars($uni['logo']) ?>" 
                                                     alt="<?= htmlspecialchars($uni['name']) ?>" class="img-fluid mb-3" style="max-height: 60px;">
                                                <h5><?= htmlspecialchars($uni['name']) ?></h5>
                                                <span class="badge badge-<?= 
                                                    $uni['status'] == 'Admitted' ? 'success' : 
                                                    ($uni['status'] == 'Rejected' ? 'danger' : 'warning') ?>">
                                                    <?= htmlspecialchars($uni['status']) ?>
                                                </span>
                                            </div>
                                            <div class="card-footer bg-transparent">
                                                <a href="applications.php" class="btn btn-sm btn-block btn-outline-primary">View Details</a>
                                            </div>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <div class="col-12">
                                    <div class="alert alert-info">
                                        You haven't applied to any universities yet. <a href="applications.php" class="alert-link">Start your application</a>.
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Quick Links -->
        <div class="row mt-4">
            <div class="col-md-4 mb-4">
                <div class="card shadow-sm h-100">
                    <div class="card-body text-center">
                        <i class="fas fa-file-upload fa-3x text-primary mb-3"></i>
                        <h5>Document Center</h5>
                        <p>Upload required documents for your applications</p>
                        <a href="documents.php" class="btn btn-primary">Go to Documents</a>
                    </div>
                </div>
            </div>
            <div class="col-md-4 mb-4">
                <div class="card shadow-sm h-100">
                    <div class="card-body text-center">
                        <i class="fas fa-comments fa-3x text-primary mb-3"></i>
                        <h5>Message Counselor</h5>
                        <p>Get answers to your questions from our experts</p>
                        <a href="messages.php" class="btn btn-primary">Send Message</a>
                    </div>
                </div>
            </div>
            <div class="col-md-4 mb-4">
                <div class="card shadow-sm h-100">
                    <div class="card-body text-center">
                        <i class="fas fa-calendar-check fa-3x text-primary mb-3"></i>
                        <h5>Book Appointment</h5>
                        <p>Schedule a meeting with your education counselor</p>
                        <a href="appointments.php" class="btn btn-primary">Book Now</a>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Footer (matches your existing site) -->
    <footer class="bg-dark text-white-50 pt-5 pb-4">
        <div class="container text-center">
            <p class="mb-0">JV Overseas Education &copy; <?= date('Y') ?>. All Rights Reserved.</p>
        </div>
    </footer>

    <!-- Back to Top -->
    <a href="#" class="btn btn-lg btn-primary rounded-0 btn-lg-square back-to-top"><i class="fa fa-angle-double-up"></i></a>

    <!-- JavaScript Libraries -->
    <script src="https://code.jquery.com/jquery-3.4.1.min.js"></script>
    <script src="https://stackpath.bootstrapcdn.com/bootstrap/4.4.1/js/bootstrap.bundle.min.js"></script>
    <script src="../../lib/easing/easing.min.js"></script>
    <script src="../../lib/waypoints/waypoints.min.js"></script>
    
    <!-- Dashboard-specific JS -->
    <script src="../../js/dashboard.js"></script>
</body>
</html>