<?php
session_start();
require_once 'config.php';

// Authentication check
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] !== 'student') {
    header('Location: student-login.php');
    exit;
}

// Get student applications
try {
    $stmt = $pdo->prepare("SELECT a.*, p.name as program, u.name as university, u.logo as university_logo
                          FROM applications a
                          JOIN programs p ON a.program_id = p.id
                          JOIN universities u ON p.university_id = u.id
                          WHERE a.student_id = ?
                          ORDER BY a.created_at DESC");
    $stmt->execute([$_SESSION['user_id']]);
    $applications = $stmt->fetchAll();
} catch (PDOException $e) {
    die("Error fetching applications: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title>Applications - JV Overseas Education</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link href="https://fonts.googleapis.com/css2?family=Jost:wght@500;600;700&family=Open+Sans:wght@400;600&display=swap" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.10.0/css/all.min.css" rel="stylesheet">
    <link href="css/style.css" rel="stylesheet">
    <style>
        .application-card {
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        .application-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 20px rgba(0,0,0,0.1);
        }
        .badge-success {
            background-color: #28a745;
        }
        .badge-info {
            background-color: #17a2b8;
        }
        .badge-warning {
            background-color: #ffc107;
            color: #212529;
        }
        .badge-danger {
            background-color: #dc3545;
        }
        .university-logo {
            height: 60px;
            object-fit: contain;
            max-width: 100%;
            padding: 10px;
            background: white;
            border-radius: 8px;
        }
    </style>
</head>
<body>

<?php include 'student-navbar.php'; ?>

<div class="container py-5">
    <div class="section-title mb-4">
        <h1 class="display-5">My Applications</h1>
        <p class="lead">Track your university applications</p>
    </div>
    
    <?php if (empty($applications)): ?>
        <div class="text-center py-5">
            <i class="fas fa-file-alt fa-4x text-muted mb-4"></i>
            <h4 class="mb-3">No applications found</h4>
            <p class="text-muted mb-4">You haven't applied to any universities yet.</p>
            <a href="apply.php" class="btn btn-primary btn-lg">Start New Application</a>
        </div>
    <?php else: ?>
        <div class="row">
            <?php foreach ($applications as $app): ?>
            <div class="col-lg-6 mb-4">
                <div class="card h-100 application-card">
                    <div class="p-3 bg-light text-center">
                        <img src="<?= htmlspecialchars($app['university_logo'] ?: 'img/university-placeholder.png') ?>" 
                             alt="<?= htmlspecialchars($app['university']) ?>" 
                             class="university-logo">
                    </div>
                    <div class="card-body">
                        <div class="d-flex justify-content-between align-items-start">
                            <div>
                                <h5 class="card-title mb-1"><?= htmlspecialchars($app['university']) ?></h5>
                                <p class="card-text text-muted mb-2"><?= htmlspecialchars($app['program']) ?></p>
                            </div>
                            <span class="badge badge-<?php 
                                if ($app['application_status'] === 'accepted') {
                                    echo 'success';
                                } elseif ($app['application_status'] === 'submitted') {
                                    echo 'info';
                                } elseif ($app['application_status'] === 'rejected') {
                                    echo 'danger';
                                } else {
                                    echo 'warning';
                                }
                            ?>">
                                <?= ucfirst(htmlspecialchars($app['application_status'])) ?>
                            </span>
                        </div>
                        
                        <div class="mt-3">
                            <div class="d-flex justify-content-between text-muted small">
                                <span>
                                    <i class="far fa-calendar-alt mr-1"></i>
                                    <?= date('M j, Y', strtotime($app['submission_date'])) ?>
                                </span>
                                <?php if ($app['application_status'] === 'accepted'): ?>
                                    <span class="text-success">
                                        <i class="far fa-check-circle mr-1"></i>
                                        Deadline: <?= date('M j, Y', strtotime($app['decision_deadline'])) ?>
                                    </span>
                                <?php endif; ?>
                            </div>
                        </div>
                        
                        <div class="mt-4 d-flex justify-content-between">
                            <a href="application-details.php?id=<?= $app['id'] ?>" class="btn btn-outline-primary">
                                View Details
                            </a>
                            <?php if ($app['application_status'] === 'accepted'): ?>
                                <a href="#" class="btn btn-success">
                                    Accept Offer
                                </a>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>
</div>

<?php include 'footer.php'; ?>

<script src="https://code.jquery.com/jquery-3.4.1.min.js"></script>
<script src="https://stackpath.bootstrapcdn.com/bootstrap/4.4.1/js/bootstrap.bundle.min.js"></script>
</body>
</html>