<?php
session_start();

// Database connection
$host = 'localhost';
$dbname = 'jv_overseas';
$username = 'root';
$password = '';

try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    die("Connection failed: " . $e->getMessage());
}

// Get university ID from URL
$university_id = $_GET['id'] ?? null;

if (!$university_id) {
    die("University ID is required.");
}

// Get university data
$stmt = $pdo->prepare("SELECT u.*, c.name as country_name, c.flag_path
                      FROM universities u
                      JOIN countries c ON u.country_id = c.id
                      WHERE u.id = ?");
$stmt->execute([$university_id]);
$university = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$university) {
    die("University not found.");
}

// Get programs for this university
$stmt = $pdo->prepare("SELECT p.*, 
                      CASE 
                          WHEN p.degree_level = 'bachelors' THEN 'Bachelor\'s'
                          WHEN p.degree_level = 'masters' THEN 'Master\'s'
                          ELSE UPPER(LEFT(p.degree_level, 1)) || LOWER(SUBSTRING(p.degree_level, 2))
                      END as degree_level_display
                      FROM programs p
                      WHERE p.university_id = ?
                      ORDER BY p.degree_level, p.name");
$stmt->execute([$university_id]);
$programs = $stmt->fetchAll();

// Get application deadlines
$application_deadlines = json_decode($university['application_deadlines'] ?? '[]', true);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="utf-8">
    <title><?= htmlspecialchars($university['name']) ?> - JV Overseas Education</title>
    <meta content="width=device-width, initial-scale=1.0" name="viewport">
    <link rel="preconnect" href="https://fonts.gstatic.com">
    <link href="https://fonts.googleapis.com/css2?family=Jost:wght@500;600;700&family=Open+Sans:wght@400;600&display=swap" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.10.0/css/all.min.css" rel="stylesheet">
    <link href="lib/owlcarousel/assets/owl.carousel.min.css" rel="stylesheet">
    <link href="css/style.css" rel="stylesheet">
    <style>
        .university-header {
            background: linear-gradient(rgba(0, 0, 0, 0.5), rgba(0, 0, 0, 0.5)), 
                        url('<?= htmlspecialchars($university['logo'] ?: 'img/university-placeholder.png') ?>');
            background-size: cover;
            background-position: center;
            color: white;
            padding: 80px 0;
            margin-bottom: 30px;
        }
        .university-logo-large {
            width: 150px;
            height: 150px;
            object-fit: contain;
            background: white;
            padding: 15px;
            border-radius: 10px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        }
        .partner-badge {
            font-size: 0.9em;
            padding: 5px 12px;
            border-radius: 20px;
            font-weight: 600;
        }
        .partner-basic {
            background: #f8f9fa;
            color: #6c757d;
        }
        .partner-preferred {
            background: #fff3cd;
            color: #856404;
        }
        .partner-premium {
            background: #d1ecf1;
            color: #0c5460;
        }
        .program-card {
            border: 1px solid #eee;
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 20px;
            transition: all 0.3s;
        }
        .program-card:hover {
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            transform: translateY(-5px);
        }
        .degree-badge {
            font-size: 0.9em;
            padding: 5px 12px;
            border-radius: 20px;
            font-weight: 600;
        }
        .degree-bachelors {
            background: #d1ecf1;
            color: #0c5460;
        }
        .degree-masters {
            background: #d4edda;
            color: #155724;
        }
        .degree-phd {
            background: #f8d7da;
            color: #721c24;
        }
        .deadline-item {
            padding: 10px;
            border-bottom: 1px solid #eee;
        }
        .deadline-item:last-child {
            border-bottom: none;
        }
    </style>
</head>
<body>

<?php include 'navbar.php'; ?>

<!-- University Header -->
<div class="university-header">
    <div class="container">
        <div class="row align-items-center">
            <div class="col-lg-8">
                <h1 class="display-4"><?= htmlspecialchars($university['name']) ?></h1>
                <div class="d-flex align-items-center mb-3">
                    <img src="img/flags/<?= strtolower($university['country_name']) ?>.png" 
                         alt="<?= htmlspecialchars($university['country_name']) ?>" class="country-flag">
                    <span><?= htmlspecialchars($university['city'] ?? $university['country_name']) ?></span>
                    <?php if ($university['ranking']): ?>
                    <span class="ml-3">#<?= $university['ranking'] ?> Worldwide</span>
                    <?php endif; ?>
                </div>
                <div class="d-flex align-items-center">
                    <span class="partner-badge partner-<?= strtolower($university['partnership_level'] ?: 'basic') ?>">
                        <?= ucfirst($university['partnership_level'] ?: 'Basic') ?> Partner
                    </span>
                    <?php if ($university['application_fee']): ?>
                    <span class="ml-3">Application Fee: $<?= number_format($university['application_fee'], 2) ?></span>
                    <?php endif; ?>
                </div>
            </div>
            <div class="col-lg-4 text-center">
                <img src="<?= htmlspecialchars($university['logo'] ?: 'img/university-placeholder.png') ?>" 
                     alt="<?= htmlspecialchars($university['name']) ?>" class="university-logo-large">
            </div>
        </div>
    </div>
</div>

<!-- University Content -->
<div class="container py-5">
    <div class="row">
        <div class="col-lg-8">
            <!-- About Section -->
            <div class="mb-5">
                <h2>About <?= htmlspecialchars($university['name']) ?></h2>
                <p><?= nl2br(htmlspecialchars($university['description'])) ?></p>
                
                <?php if ($university['website']): ?>
                <div class="mt-4">
                    <a href="<?= htmlspecialchars($university['website']) ?>" target="_blank" class="btn btn-primary">
                        Visit University Website <i class="fas fa-external-link-alt"></i>
                    </a>
                </div>
                <?php endif; ?>
            </div>
            
            <!-- Programs Section -->
            <div class="mb-5">
                <h2>Available Programs</h2>
                <?php if (empty($programs)): ?>
                <div class="text-center py-5">
                    <i class="fas fa-graduation-cap fa-3x text-muted mb-3"></i>
                    <p class="text-muted">No programs currently available at this university.</p>
                </div>
                <?php else: ?>
                <?php foreach ($programs as $p): ?>
                <div class="program-card">
                    <div class="d-flex justify-content-between align-items-start">
                        <div>
                            <h4><?= htmlspecialchars($p['name']) ?></h4>
                            <span class="degree-badge degree-<?= $p['degree_level'] ?>">
                                <?= htmlspecialchars($p['degree_level_display']) ?>
                            </span>
                        </div>
                        <div class="text-right">
                            <div class="fee-info">
                                <strong>$<?= number_format($p['tuition_fee'], 2) ?></strong><br>
                                per year
                            </div>
                        </div>
                    </div>
                    
                    <div class="mt-3">
                        <p class="text-muted"><?= substr(htmlspecialchars($p['program_description']), 0, 200) ?>...</p>
                        
                        <div class="d-flex justify-content-between">
                            <div>
                                <small>Duration: <?= $p['duration_months'] ?> months</small>
                            </div>
                            <div>
                                <small>Intakes: <?= htmlspecialchars(implode(', ', json_decode($p['intake_months'] ?? '[]', true) ?: [])) ?></small>
                            </div>
                        </div>
                    </div>
                    
                    <div class="mt-3">
                        <a href="apply.php?university=<?= $university_id ?>&program=<?= $p['id'] ?>" class="btn btn-primary btn-sm">
                            Apply Now
                        </a>
                        <a href="program-profile.php?id=<?= $p['id'] ?>" class="btn btn-outline-primary btn-sm">
                            View Details
                        </a>
                    </div>
                </div>
                <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
        
        <div class="col-lg-4">
            <!-- Quick Facts -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5>Quick Facts</h5>
                </div>
                <div class="card-body">
                    <ul class="list-unstyled">
                        <li class="mb-3"><i class="fas fa-map-marker-alt mr-2"></i> 
                            <strong>Location:</strong><br><?= htmlspecialchars($university['city'] ?? 'Not specified') ?>, <?= htmlspecialchars($university['country_name']) ?>
                        </li>
                        <?php if ($university['ranking']): ?>
                        <li class="mb-3"><i class="fas fa-trophy mr-2"></i> 
                            <strong>Ranking:</strong><br>#<?= $university['ranking'] ?> Worldwide
                        </li>
                        <?php endif; ?>
                        <li class="mb-3"><i class="fas fa-graduation-cap mr-2"></i> 
                            <strong>Programs:</strong><br><?= count($programs) ?> Available
                        </li>
                        <?php if ($application_deadlines): ?>
                        <li class="mb-3"><i class="fas fa-calendar-alt mr-2"></i> 
                            <strong>Application Deadlines:</strong><br>
                            <?= htmlspecialchars(implode(', ', $application_deadlines)) ?>
                        </li>
                        <?php endif; ?>
                        <?php if ($university['contact_name']): ?>
                        <li class="mb-3"><i class="fas fa-user-tie mr-2"></i> 
                            <strong>Contact:</strong><br><?= htmlspecialchars($university['contact_name']) ?>
                        </li>
                        <?php endif; ?>
                        <?php if ($university['contact_email']): ?>
                        <li class="mb-3"><i class="fas fa-envelope mr-2"></i> 
                            <strong>Email:</strong><br><?= htmlspecialchars($university['contact_email']) ?>
                        </li>
                        <?php endif; ?>
                    </ul>
                </div>
            </div>
            
            <!-- Application Deadlines -->
            <?php if ($application_deadlines): ?>
            <div class="card mb-4">
                <div class="card-header">
                    <h5>Application Deadlines</h5>
                </div>
                <div class="card-body">
                    <?php foreach ($application_deadlines as $deadline): ?>
                    <div class="deadline-item">
                        <i class="fas fa-calendar-check mr-2"></i>
                        <?= htmlspecialchars($deadline) ?> Intake
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>
            
            <!-- Contact Us -->
            <div class="card">
                <div class="card-header">
                    <h5>Interested in Applying?</h5>
                </div>
                <div class="card-body">
                    <p>Let us help you with your application to <?= htmlspecialchars($university['name']) ?>.</p>
                    <a href="contact.php" class="btn btn-primary btn-block">Get Free Consultation</a>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'footer.php'; ?>
<script src="https://code.jquery.com/jquery-3.4.1.min.js"></script>
<script src="https://stackpath.bootstrapcdn.com/bootstrap/4.4.1/js/bootstrap.bundle.min.js"></script>
<script src="lib/easing/easing.min.js"></script>
<script src="lib/owlcarousel/owl.carousel.min.js"></script>
<script src="mail/jqBootstrapValidation.min.js"></script>
<script src="mail/contact.js"></script>
<script src="js/main.js"></script>
</body>
</html>